const express = require('express');
const fs = require('fs').promises;
const path = require('path');
const AdmZip = require('adm-zip');
const { extractFull } = require('node-7z');

const app = express();
const PORT = 80;
const BASE_DIR = path.resolve(process.cwd());

// CORS設定 - 全て許可
app.use((req, res, next) => {
  res.header('Access-Control-Allow-Origin', '*');
  res.header('Access-Control-Allow-Methods', 'GET, POST, PUT, DELETE, OPTIONS');
  res.header('Access-Control-Allow-Headers', '*');
  if (req.method === 'OPTIONS') {
    return res.sendStatus(200);
  }
  next();
});

app.use(express.json({ limit: '100mb' }));

// パスが作業ディレクトリ外に出ないかチェック
function validatePath(targetPath) {
  const resolved = path.resolve(BASE_DIR, targetPath);
  if (!resolved.startsWith(BASE_DIR)) {
    throw new Error('Path traversal detected: Access denied');
  }
  return resolved;
}

// Base64プレフィックスを除去
function cleanBase64(data) {
  const match = data.match(/^data:[^;]+;base64,(.+)$/);
  return match ? match[1] : data;
}

// 1. POST /add - Base64ファイルを保存
app.post('/add', async (req, res) => {
  try {
    const { data, dir } = req.body;
    if (!data || !dir) {
      return res.status(400).json({ error: 'Missing required fields: data and dir' });
    }

    const fullPath = validatePath(dir);
    const cleanedData = cleanBase64(data);
    const buffer = Buffer.from(cleanedData, 'base64');

    await fs.mkdir(path.dirname(fullPath), { recursive: true });
    await fs.writeFile(fullPath, buffer);

    res.json({
      success: true,
      dir: path.relative(BASE_DIR, fullPath),
      size: buffer.length
    });
  } catch (err) {
    res.status(500).json({ error: `Failed to save file: ${err.message}` });
  }
});

// 2. POST /rename - ファイルをリネーム
app.post('/rename', async (req, res) => {
  try {
    const { dir, name } = req.body;
    if (!dir || !name) {
      return res.status(400).json({ error: 'Missing required fields: dir and name' });
    }

    if (name.includes('/') || name.includes('\\')) {
      return res.status(400).json({ error: 'Invalid name: Path separators not allowed' });
    }

    const oldPath = validatePath(dir);
    const newPath = path.join(path.dirname(oldPath), name);

    validatePath(path.relative(BASE_DIR, newPath));

    await fs.rename(oldPath, newPath);

    res.json({
      success: true,
      dir: path.relative(BASE_DIR, newPath)
    });
  } catch (err) {
    res.status(500).json({ error: `Failed to rename: ${err.message}` });
  }
});

// 3. POST /extract - zip/7zを解凍
app.post('/extract', async (req, res) => {
  try {
    const { dir, to } = req.body;
    if (!dir || !to) {
      return res.status(400).json({ error: 'Missing required fields: dir and to' });
    }

    const sourcePath = validatePath(dir);
    const destPath = validatePath(to);
    const ext = path.extname(sourcePath).toLowerCase();

    await fs.mkdir(destPath, { recursive: true });

    if (ext === '.zip') {
      const zip = new AdmZip(sourcePath);
      zip.extractAllTo(destPath, true);
      res.json({
        success: true,
        extracted: path.relative(BASE_DIR, destPath),
        type: 'zip'
      });
    } else if (ext === '.7z') {
      let responded = false;

      const seven = extractFull(sourcePath, destPath, {
        $bin: '7za',
        $progress: false
      });

      seven.on('end', () => {
        if (!responded) {
          responded = true;
          res.json({
            success: true,
            extracted: path.relative(BASE_DIR, destPath),
            type: '7z'
          });
        }
      });

      seven.on('error', (err) => {
        if (!responded) {
          responded = true;
          res.status(500).json({
            error: `7z extraction failed: ${err.message}`
          });
        }
      });
    } else {
      res.status(400).json({
        error: 'Unsupported file type',
        supported: ['.zip', '.7z'],
        received: ext
      });
    }
  } catch (err) {
    res.status(500).json({ error: `Failed to extract: ${err.message}` });
  }
});

// 4. POST /delete - ファイルまたはディレクトリを削除
app.post('/delete', async (req, res) => {
  try {
    const { dir } = req.body;
    if (!dir) {
      return res.status(400).json({ error: 'Missing required field: dir' });
    }

    const targetPath = validatePath(dir);

    if (targetPath === BASE_DIR) {
      return res.status(403).json({ error: 'Cannot delete base directory' });
    }

    await fs.rm(targetPath, { recursive: true, force: true });

    res.json({
      success: true,
      dir: path.relative(BASE_DIR, targetPath)
    });
  } catch (err) {
    res.status(500).json({ error: `Failed to delete: ${err.message}` });
  }
});

// 5. GET /list - 元のまま残す
app.get('/list', async (req, res) => {
  try {
    const items = [];
    const visitedPaths = new Set();

    async function traverse(currentDir) {
      let realPath;
      try {
        realPath = await fs.realpath(currentDir);
      } catch {
        return;
      }

      if (visitedPaths.has(realPath)) return;
      visitedPaths.add(realPath);

      let entries;
      try {
        entries = await fs.readdir(currentDir, { withFileTypes: true });
      } catch {
        return;
      }

      for (const entry of entries) {
        const fullPath = path.join(currentDir, entry.name);
        const relativePath = './' + path.relative(BASE_DIR, fullPath).replace(/\\/g, '/');

        items.push({
          dir: relativePath,
          name: entry.name,
          type: entry.isDirectory() ? 'directory' : 'file'
        });

        if (entry.isDirectory() && !entry.isSymbolicLink()) {
          await traverse(fullPath);
        }
      }
    }

    await traverse(BASE_DIR);
    res.json(items);
  } catch (err) {
    res.status(500).json({ error: `Failed to list files: ${err.message}` });
  }
});

// 6. GET /* → ファイルは dataURL(Base64)、ディレクトリは JSON
app.get(/(.*)/, async (req, res) => {
  try {
    const requestPath = req.params[0] || '';
    const cleanPath = requestPath.replace(/^\//, '');
    const filePath = validatePath(cleanPath);

    const stats = await fs.stat(filePath);

    // ディレクトリ → JSON
    if (stats.isDirectory()) {
      const entries = await fs.readdir(filePath, { withFileTypes: true });
      return res.json(entries.map(entry => ({
        name: entry.name,
        type: entry.isDirectory() ? 'directory' : 'file',
        dir: path.join(req.path, entry.name).replace(/\\/g, '/')
      })));
    }

    // ファイル → dataURL(Base64)
    const buffer = await fs.readFile(filePath);

    const ext = path.extname(filePath).toLowerCase();
    const mime =
      ext === '.png' ? 'image/png' :
      ext === '.jpg' || ext === '.jpeg' ? 'image/jpeg' :
      ext === '.gif' ? 'image/gif' :
      ext === '.txt' ? 'text/plain' :
      ext === '.json' ? 'application/json' :
      ext === '.html' ? 'text/html' :
      'application/octet-stream';

    const base64 = buffer.toString('base64');
    const dataURL = `data:${mime};base64,${base64}`;

    res.json({
      success: true,
      data: dataURL,
      size: buffer.length,
      dir: cleanPath
    });

  } catch (err) {
    res.status(404).json({ error: `Not found: ${err.message}` });
  }
});

app.listen(PORT, () => {
  console.log(`===========================================`);
  console.log(`File Management Server`);
  console.log(`===========================================`);
  console.log(`Server: http://localhost:${PORT}`);
  console.log(`Working directory: ${BASE_DIR}`);
  console.log(`===========================================`);
  console.log(`GET /* → dataURL(Base64) or directory JSON`);
  console.log(`POST /add`);
  console.log(`POST /rename`);
  console.log(`POST /extract`);
  console.log(`POST /delete`);
  console.log(`GET /list`);
  console.log(`===========================================`);
});
